"use client";

import { useRef } from "react";
import Image from "next/image";
import { Link } from "@/core/lib/i18n/navigation";
import { motion, useScroll, useTransform, useMotionTemplate, useMotionValue } from "motion/react";
import { ArrowLeft, ExternalLink, Calendar, Layers, Cpu, Globe } from "lucide-react";
import { cn } from "@/core/lib/utils";
import type { StrapiProject, StrapiMedia, StrapiBlockNode } from "@/core/lib/api/types";
import { getStrapiMediaUrl } from "@/core/lib/api/client";
import { MarkdownRenderer } from "@/components/markdown-renderer";
import { BlocksRenderer } from "@/modules/blog/components";

// LightGallery
import LightGallery from "lightgallery/react";
import lgThumbnail from "lightgallery/plugins/thumbnail";
import lgZoom from "lightgallery/plugins/zoom";

import "lightgallery/css/lightgallery.css";
import "lightgallery/css/lg-zoom.css";
import "lightgallery/css/lg-thumbnail.css";

// --- Types ---
interface ProjectDetailsContentProps {
    project: StrapiProject;
}

// --- Gallery Card Component ---
function GalleryCard({
    media,
    index,
    onClick,
    className
}: {
    media: StrapiMedia;
    index: number;
    onClick: () => void;
    className?: string;
}) {
    const mouseX = useMotionValue(0);
    const mouseY = useMotionValue(0);

    const handleMouseMove = (event: React.MouseEvent<HTMLDivElement>) => {
        const { currentTarget, clientX, clientY } = event;
        const { left, top } = currentTarget.getBoundingClientRect();
        mouseX.set(clientX - left);
        mouseY.set(clientY - top);
    };

    return (
        <motion.div
            onClick={onClick}
            onMouseMove={handleMouseMove}
            className={cn(
                "group/spotlight group relative h-full flex-shrink-0 cursor-pointer overflow-hidden rounded-3xl border border-slate-800 bg-slate-950/80 transition-all duration-500 hover:border-slate-600/50",
                className
            )}
            initial="initial"
            whileInView="visible"
            whileHover="hover"
            viewport={{ once: true, amount: 0.1 }}
            variants={{
                initial: { opacity: 0, y: 20 },
                visible: {
                    opacity: 1,
                    y: 0,
                    transition: { delay: 0.1, duration: 0.5 }
                },
                hover: {
                    y: -4,
                    transition: { duration: 0.2 }
                }
            }}
        >
            {/* Spotlight Effect */}
            <motion.div
                className="pointer-events-none absolute -inset-px z-5 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
                style={{
                    background: useMotionTemplate`
            radial-gradient(
              450px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.12),
              transparent 80%
            )
          `,
                }}
            />

            <div className="absolute inset-0 z-0">
                <Image
                    src={getStrapiMediaUrl(media.url)}
                    alt={media.alternativeText || `Gallery image ${index + 1}`}
                    fill
                    className="object-cover transition-transform duration-700 ease-in-out group-hover:scale-105"
                    sizes="(max-width: 768px) 100vw, 800px"
                />
            </div>

            {/* Overlay */}
            <div className="absolute inset-0 z-10 pointer-events-none bg-gradient-to-t from-slate-950/90 via-transparent to-transparent opacity-60 transition-opacity duration-500 group-hover:opacity-40" />

            {/* Content */}
            <div className="relative z-20 h-full flex flex-col justify-end p-6 pointer-events-none">
                <div className="transform translate-y-4 opacity-0 transition-all duration-300 group-hover:translate-y-0 group-hover:opacity-100 flex justify-between items-end">
                    <span className="text-white font-medium drop-shadow-md">{media.name}</span>
                    <span className="inline-flex items-center justify-center w-10 h-10 rounded-full bg-white/20 backdrop-blur-md text-white hover:bg-white hover:text-black transition-colors">
                        <ExternalLink className="w-5 h-5" />
                    </span>
                </div>
            </div>
        </motion.div>
    );
}

// --- Project Gallery Component ---
function ProjectGallery({ project }: { project: StrapiProject }) {
    const containerRef = useRef<HTMLDivElement>(null);
    const galleryRef = useRef<any>(null);

    const galleryImages = project.gallery || [];
    const scrollHeight = `${Math.max(300, galleryImages.length * 80)}vh`;

    const { scrollYProgress } = useScroll({
        target: containerRef,
    });

    const x = useTransform(scrollYProgress, [0, 1], ["0%", "-85%"]);

    const openLightbox = (index: number) => {
        if (galleryRef.current) {
            galleryRef.current.openGallery(index);
        }
    };

    const galleryItems = galleryImages.map(img => ({
        id: img.documentId || img.url,
        src: getStrapiMediaUrl(img.url),
        thumb: getStrapiMediaUrl(img.url),
        subHtml: `<div class="lightgallery-caption"><h4>${img.alternativeText || img.name || project.title}</h4></div>`,
    }));

    return (
        <section id="gallery" className="relative bg-slate-950 pt-20">
            {/* Header for gallery */}
            <div className="container px-4 md:px-6 mb-12 lg:mb-0 relative z-20">
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                >
                    <h2 className="text-3xl md:text-5xl font-bold text-white mb-4">Project Gallery</h2>
                    <p className="text-slate-400 max-w-xl text-lg">Visual breakdown of the interface and components.</p>
                </motion.div>
            </div>

            {/* LightGallery Instance (Invisible) */}
            <div className="hidden">
                <LightGallery
                    dynamic
                    dynamicEl={galleryItems}
                    onInit={(detail) => {
                        galleryRef.current = detail.instance;
                    }}
                    plugins={[lgThumbnail, lgZoom]}
                    elementClassNames="hidden"
                />
            </div>

            {/* Desktop Horizontal Scroll */}
            <div ref={containerRef} className="hidden lg:block relative z-10" style={{ height: scrollHeight }}>
                <div className="sticky top-0 flex h-screen items-center overflow-hidden">
                    <motion.div style={{ x }} className="flex gap-12 px-12 md:px-24 items-center">
                        {/* Intro spacer */}
                        <div className="w-[10vw] flex-shrink-0" />

                        {galleryImages.map((media, i) => (
                            <div key={i} className="w-[60vw] max-w-[900px] h-[65vh] flex-shrink-0 relative">
                                <GalleryCard
                                    media={media}
                                    index={i}
                                    onClick={() => openLightbox(i)}
                                    className="w-full h-full shadow-2xl shadow-blue-900/10"
                                />
                                {/* Number indicator */}
                                <div className="absolute -bottom-10 left-10 text-6xl font-black text-slate-800/50 pointer-events-none select-none">
                                    {(i + 1).toString().padStart(2, '0')}
                                </div>
                            </div>
                        ))}

                        {/* Outro spacer */}
                        <div className="w-[10vw] flex-shrink-0" />
                    </motion.div>
                </div>
            </div>

            {/* Mobile Grid View */}
            <div className="lg:hidden container px-4 pb-24 grid grid-cols-1 gap-8">
                {galleryImages.map((media, i) => (
                    <div key={i} className="aspect-video relative h-full min-h-[300px]">
                        <GalleryCard
                            media={media}
                            index={i}
                            onClick={() => openLightbox(i)}
                            className="w-full h-full"
                        />
                    </div>
                ))}
            </div>
        </section>
    );
}


export function ProjectDetailsContent({ project }: ProjectDetailsContentProps) {
    console.log("Project Details Data:", JSON.stringify(project, null, 2));
    const hasGallery = project.gallery && project.gallery.length > 0;

    if (!project) return null;

    return (
        <div className="bg-slate-950 text-slate-50 min-h-screen">
            {/* --- Hero Section --- */}
            <section className="relative h-[85vh] min-h-[600px] flex items-end justify-start overflow-hidden">
                {/* Background Image Parallax Effect */}
                <div className="absolute inset-0 z-0 select-none">
                    {project.cover_image && (
                        <div className="relative w-full h-full">
                            <Image
                                src={getStrapiMediaUrl(project.cover_image.url)}
                                alt={project.title}
                                fill
                                className="object-cover"
                                priority
                            />
                            <div className="absolute inset-0 bg-gradient-to-t from-slate-950 via-slate-950/70 to-slate-950/30" />
                            <div className="absolute inset-0 bg-slate-950/20 backdrop-blur-[1px]" />
                        </div>
                    )}
                </div>

                <div className="container relative z-10 pb-24 px-4 md:px-6">
                    <Link
                        href="/#portfolio"
                        className="inline-flex items-center gap-2 text-slate-400 hover:text-white mb-8 transition-colors group opacity-80 hover:opacity-100"
                    >
                        <div className="p-2 rounded-full bg-white/5 border border-white/10 group-hover:bg-white/10 transition-colors">
                            <ArrowLeft className="w-4 h-4 group-hover:-translate-x-0.5 transition-transform" />
                        </div>
                        <span className="text-sm font-medium">Back to Projects</span>
                    </Link>

                    <motion.h1
                        initial={{ opacity: 0, y: 30 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.7, ease: [0.22, 1, 0.36, 1] }}
                        className="text-5xl md:text-7xl lg:text-8xl font-bold tracking-tight text-white mb-8 drop-shadow-xl"
                    >
                        {project.title}
                    </motion.h1>

                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.7, delay: 0.2, ease: [0.22, 1, 0.36, 1] }}
                        className="flex flex-wrap gap-4 md:gap-8 items-center text-slate-300"
                    >
                        {project.service?.name && (
                            <span className="flex items-center gap-2 px-4 py-2 rounded-full bg-blue-500/10 border border-blue-500/20 text-blue-300 text-sm font-medium backdrop-blur-md">
                                <Layers className="w-4 h-4" />
                                {project.service.name}
                            </span>
                        )}
                        {project.publishedAt && (
                            <span className="flex items-center gap-2 text-sm font-medium bg-black/20 px-3 py-1 rounded-full backdrop-blur-md">
                                <Calendar className="w-4 h-4 text-slate-400" />
                                {new Date(project.publishedAt).toLocaleDateString()}
                            </span>
                        )}
                        {project.live_url && (
                            <a
                                href={project.live_url}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="flex items-center gap-2 px-4 py-2 rounded-full bg-emerald-500/10 border border-emerald-500/20 text-emerald-300 text-sm font-medium hover:bg-emerald-500/20 transition-colors backdrop-blur-md"
                            >
                                <Globe className="w-4 h-4" />
                                Visit Live Site
                            </a>
                        )}
                    </motion.div>
                </div>

                {/* Scroll Indicator */}
                <motion.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1, y: [0, 10, 0] }}
                    transition={{ delay: 1, duration: 2, repeat: Infinity }}
                    className="absolute bottom-10 left-1/2 -translate-x-1/2 flex flex-col items-center gap-2 text-slate-500"
                >
                    <span className="text-xs uppercase tracking-widest opacity-60">Scroll</span>
                    <div className="w-[1px] h-12 bg-gradient-to-b from-transparent via-slate-500 to-transparent" />
                </motion.div>
            </section>


            {/* --- Overview Section --- */}
            <section className="py-24 px-4 md:px-6 relative z-10">
                <div className="container max-w-7xl mx-auto grid grid-cols-1 lg:grid-cols-12 gap-12 lg:gap-24">
                    {/* Left: Content */}
                    <div className="lg:col-span-8">
                        <motion.div
                            initial={{ opacity: 0, y: 20 }}
                            whileInView={{ opacity: 1, y: 0 }}
                            viewport={{ once: true }}
                            transition={{ duration: 0.5 }}
                        >
                            <h2 className="text-sm font-bold text-blue-500 tracking-widest uppercase mb-8 flex items-center gap-3">
                                <span className="w-8 h-[2px] bg-blue-500"></span>
                                Methods & Journey
                            </h2>
                            {typeof project.overview === 'string' ? (
                                <MarkdownRenderer content={project.overview} />
                            ) : Array.isArray(project.overview) ? (
                                <BlocksRenderer content={project.overview as StrapiBlockNode[]} />
                            ) : (
                                <p className="text-slate-500 italic text-lg text-center py-10">Project description coming soon...</p>
                            )}
                        </motion.div>
                    </div>

                    {/* Right: Sidebar Info */}
                    <div className="lg:col-span-4 space-y-10">
                        <motion.div
                            initial={{ opacity: 0, x: 20 }}
                            whileInView={{ opacity: 1, x: 0 }}
                            viewport={{ once: true }}
                            transition={{ duration: 0.5, delay: 0.2 }}
                            className="p-8 rounded-3xl border border-slate-800 bg-slate-900/60 backdrop-blur-md sticky top-24"
                        >
                            <h3 className="text-xl font-semibold text-white mb-6 flex items-center gap-2">
                                <Cpu className="w-5 h-5 text-purple-400" />
                                Technologies
                            </h3>
                            <div className="flex flex-wrap gap-2 mb-8">
                                {project.technologies?.map((tech) => (
                                    <div key={tech.documentId} className="group relative flex items-center gap-2 px-3 py-2.5 rounded-xl bg-slate-800/50 border border-slate-700/50 hover:border-purple-500/30 hover:bg-purple-500/10 transition-all duration-300">
                                        {tech.logo && (
                                            <div className="relative w-5 h-5">
                                                <Image
                                                    src={getStrapiMediaUrl(tech.logo.url)}
                                                    alt={tech.title || ""}
                                                    fill
                                                    className="object-contain grayscale group-hover:grayscale-0 transition-all duration-300"
                                                />
                                            </div>
                                        )}
                                        <span className="text-sm font-medium text-slate-300 group-hover:text-purple-300 transition-colors">
                                            {tech.title}
                                        </span>
                                    </div>
                                ))}
                                {(!project.technologies || project.technologies.length === 0) && (
                                    <span className="text-slate-500 text-sm italic">Not specified</span>
                                )}
                            </div>

                            <div className="w-full h-px bg-slate-800 my-8" />

                            <h3 className="text-xl font-semibold text-white mb-6 flex items-center gap-2">
                                <Globe className="w-5 h-5 text-emerald-400" />
                                Details
                            </h3>
                            <ul className="space-y-4">
                                {project.categories?.map((cat) => (
                                    <li key={cat.documentId} className="flex items-center justify-between text-sm py-2 border-b border-slate-800 last:border-0">
                                        <span className="text-slate-400">Category</span>
                                        <span className="text-emerald-300 font-medium bg-emerald-500/10 px-2 py-1 rounded">{cat.name}</span>
                                    </li>
                                ))}
                                <li className="flex items-center justify-between text-sm py-2 border-b border-slate-800 last:border-0">
                                    <span className="text-slate-400">Service</span>
                                    <span className="text-white font-medium">{project.service?.name || "-"}</span>
                                </li>
                            </ul>
                        </motion.div>
                    </div>
                </div>
            </section>

            {/* --- Gallery Section --- */}
            {hasGallery && <ProjectGallery project={project} />}

            {/* --- Next Project CTA --- */}
            <section className="py-32 border-t border-slate-900 relative overflow-hidden bg-slate-950">
                <div className="absolute inset-0 bg-blue-600/5" />
                <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[600px] h-[600px] bg-blue-500/10 rounded-full blur-[120px] pointer-events-none" />

                <div className="container relative z-10 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                    >
                        <p className="text-slate-400 mb-8 font-medium tracking-widest uppercase text-sm">Ready for the next one?</p>
                        <Link
                            href="/#portfolio"
                            className="inline-flex items-center gap-4 text-4xl md:text-6xl font-bold text-white hover:text-transparent hover:bg-clip-text hover:bg-gradient-to-r hover:from-blue-400 hover:to-emerald-400 transition-all duration-300 group"
                        >
                            View All Projects
                            <ExternalLink className="w-8 h-8 md:w-16 md:h-16 group-hover:translate-x-3 group-hover:-translate-y-3 transition-transform duration-300 text-white" />
                        </Link>
                    </motion.div>
                </div>
            </section>
        </div>
    );
}
